#include <iostream>
#include <limits>
#include <string>

using namespace std;

/**
 * @brief Υπολογίζει το συνολικό επίδομα παιδιών βάσει των δεδομένων κανόνων.
 * * @param num_children Ο αριθμός των παιδιών (ακέραιος).
 * @return float Το συνολικό επίδομα σε ευρώ.
 */
float calculate_child_benefit(int num_children) {
    // Το επίδομα για 0 ή αρνητικό αριθμό παιδιών είναι 0
    if (num_children <= 0) {
        return 0.0f;
    }

    float total_benefit = 0.0f;
    int remaining_children = num_children;

    // 1. Για τα 2 πρώτα παιδιά: 18 €/παιδί
    if (remaining_children >= 2) {
        total_benefit += 2 * 18.0f;
        remaining_children -= 2;
    } else {
        // Αν είναι 1 παιδί
        total_benefit += remaining_children * 18.0f;
        remaining_children = 0;
    }

    // 2. Για το τρίτο παιδί: 35 €
    if (remaining_children >= 1) {
        total_benefit += 35.0f;
        remaining_children -= 1;
    }

    // 3. Για το τέταρτο παιδί: 47 €
    if (remaining_children >= 1) {
        total_benefit += 47.0f;
        remaining_children -= 1;
    }

    // 4. Για κάθε επιπλέον παιδί μετά το τέταρτο (από το 5ο και μετά): 73 €
    if (remaining_children > 0) {
        total_benefit += remaining_children * 73.0f;
    }

    return total_benefit;
}

// -------------------------------------------------------------

int main() {
    const int MAX_CHILDREN = 12;

    while (true) {
        string user_input;
        cout << "Δώστε τον αριθμό των παιδιών: ";
        
        // Λήψη της εισόδου ως string για να ελεγχθούν οι ειδικοί χαρακτήρες
        if (!(cin >> user_input)) {
            // Αυτό συμβαίνει αν η είσοδος κλείσει (EOF)
            break; 
        }

        // Ελεγχος για ειδικούς χαρακτήρες (-1, q, =)
        if (user_input == "-1" || user_input == "q" || user_input == "=") {
            cout << "Το συνολικό επίδομα είναι 0 €" << endl;
            // Στην C++ είναι πιο πρακτικό να συνεχίσουμε τον βρόχο ή να τερματίσουμε
            continue; 
            // Αν θέλαμε να τερματίσει: return 0;
        }

        try {
            // Μετατροπή της συμβολοσειράς σε ακέραιο
            size_t pos;
            int num_children = stoi(user_input, &pos);
            
            // Έλεγχος αν ολόκληρη η συμβολοσειρά μετατράπηκε σε αριθμό
            if (pos != user_input.length()) {
                throw std::invalid_argument("Μη αριθμητική είσοδος.");
            }

            // Έλεγχος για έγκυρο εύρος (0-12)
            if (num_children >= 0 && num_children <= MAX_CHILDREN) {
                
                float total_benefit = calculate_child_benefit(num_children);
                
                // Εκτύπωση αποτελέσματος. Χρησιμοποιούμε int() για να εμφανίζεται 
                // ο ακέραιος αριθμός (118) αντί του float (118.0)
                cout << "Το συνολικό επίδομα είναι " << (int)total_benefit << " €" << endl;
                
                // Έγκυρη είσοδος, το πρόγραμμα τερματίζει
                break; 

            } else {
                cout << "Ο αριθμός των παιδιών πρέπει να είναι μεταξύ 0 και " << MAX_CHILDREN << ". Δοκιμάστε ξανά." << endl;
            }

        } catch (const std::invalid_argument& e) {
            // Πιάνουμε το σφάλμα αν η είσοδος δεν ήταν έγκυρος ακέραιος (π.χ. 'a', "12.5")
            // ή αν η stoi απέτυχε
            cout << "Λάθος. Η είσοδος πρέπει να είναι ακέραιος αριθμός." << endl;
            
        } catch (const std::out_of_range& e) {
             // Πιάνουμε το σφάλμα αν ο αριθμός είναι πολύ μεγάλος για int
             cout << "Λάθος. Ο αριθμός είναι πολύ μεγάλος." << endl;
        }
    }

    return 0;
}
